<?php
/**
 * Plugin Name: Lana Downloads Manager - PDF Viewer Addon
 * Plugin URI: http://lana.codes/lana-product/lana-downloads-manager/
 * Description: Add PDF Viewer to Lana Downloads Manager.
 * Version: 1.0.0
 * Author: Lana Codes
 * Author URI: http://lana.codes/
 * Text Domain: lana-downloads-manager-pdf-viewer-addon
 * Domain Path: /languages
 */

defined( 'ABSPATH' ) or die();
define( 'LANA_DOWNLOADS_MANAGER_PDF_VIEWER_ADDON_VERSION', '1.0.0' );
define( 'LANA_DOWNLOADS_MANAGER_PDF_VIEWER_ADDON_DIR_URL', plugin_dir_url( __FILE__ ) );
define( 'LANA_DOWNLOADS_MANAGER_PDF_VIEWER_ADDON_DIR_PATH', plugin_dir_path( __FILE__ ) );

/**
 * Language
 * load
 */
load_plugin_textdomain( 'lana-downloads-manager-pdf-viewer-addon', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );

/**
 * Lana Downloads Manager
 * required
 */
function lana_downloads_manager_pdf_viewer_addon_plugin_activate() {

	if ( ! is_plugin_active( 'lana-downloads-manager/lana-downloads-manager.php' ) && current_user_can( 'activate_plugins' ) ) {
		wp_die( __( 'Sorry, but this plugin requires the Lana Downloads Manager Plugin to be installed and active.', 'lana-downloads-manager-pdf-viewer-addon' ) );
	}
}

register_activation_hook( __FILE__, 'lana_downloads_manager_pdf_viewer_addon_plugin_activate' );

/**
 * Lana Downloads Manager
 * view pdf instead of download
 *
 * @param $download_id
 */
function lana_downloads_manager_pdf_viewer_addon_view_pdf_instead_of_download( $download_id ) {

	$file_url = get_post_meta( $download_id, 'lana_download_file_url', true );

	if ( empty( $file_url ) ) {
		return;
	}

	list( $file_path, $remote_file, $local_file ) = lana_downloads_manager_parse_file_path( $file_url );

	if ( empty( $file_path ) ) {
		return;
	}

	/**
	 * Check pdf
	 */
	$file_ext = pathinfo( wp_basename( $file_path ), PATHINFO_EXTENSION );

	if ( 'pdf' != $file_ext ) {
		return;
	}

	/**
	 * Counter
	 */
	$lana_downloads_manager_counter = get_option( 'lana_downloads_manager_counter', true );
	$increment_download_count       = false;

	/**
	 * Check counter is enabled?
	 */
	if ( $lana_downloads_manager_counter ) {

		/**
		 * Check Cookie
		 */
		if ( lana_downloads_manager_cookie_exists( $download_id ) == false ) {
			$increment_download_count = true;
			lana_downloads_manager_set_cookie( $download_id );
		}

		/**
		 * Check Log
		 */
		if ( lana_downloads_manager_get_log_user_ip_has_downloaded( $download_id ) ) {
			$increment_download_count = false;
		}
	}

	/**
	 * Local file
	 */
	if ( $file_path && $local_file ) {
		lana_downloads_manager_add_log( $download_id );
		lana_downloads_manager_add_download_count( $download_id, $increment_download_count );

		$filename = wp_basename( parse_url( $file_path, PHP_URL_PATH ) );

		header( 'Content-Description: PDF View' );
		header( 'Content-Type: application/pdf' );
		header( 'Content-Disposition: inline; filename="' . $filename . '"' );
		header( 'Content-Length:' . filesize( $file_path ) );
		header( 'Connection: Keep-Alive' );
		header( 'Expires: 0' );
		header( 'Cache-Control: no-store, must-revalidate' );
		header( 'Pragma: no-cache' );
		readfile( $file_path );
		exit;
	}

	/**
	 * Remote file
	 */
	if ( $file_path && $remote_file ) {
		lana_downloads_manager_add_log( $download_id );
		lana_downloads_manager_add_download_count( $download_id, $increment_download_count );

		$filename = wp_basename( parse_url( $file_path, PHP_URL_PATH ) );
		$file_ext = pathinfo( wp_basename( $file_path ), PATHINFO_EXTENSION );

		header( 'Content-Description: PDF View' );
		header( 'Content-Type: application/pdf' );
		header( 'Content-Disposition: inline; filename="' . $filename . '"' );
		header( 'Connection: Keep-Alive' );
		header( 'Expires: 0' );
		header( 'Cache-Control: no-cache, no-store, must-revalidate' );
		header( 'Pragma: no-cache' );

		/** file without extension */
		if ( empty( $file_ext ) ) {

			/** check remote file header content disposition */
			if ( function_exists( 'get_headers' ) ) {
				$file_headers = get_headers( $file_path, 1 );

				if ( ! empty( $file_headers['Content-Disposition'] ) ) {
					header( 'Content-Disposition: ' . $file_headers['Content-Disposition'] );
				}
			}
		}

		readfile( $file_path );
		exit;
	}
}

add_action( 'lana_downloads_manager_before_file_download', 'lana_downloads_manager_pdf_viewer_addon_view_pdf_instead_of_download' );

/**
 * Lana Download Shortcode
 * with Bootstrap
 *
 * @param $atts
 *
 * @return string
 */
function lana_view_pdf_instead_of_download_shortcode( $atts ) {
	$a = shortcode_atts( array(
		'id'      => '',
		'file'    => '',
		'text'    => __( 'View', 'lana-downloads-manager-pdf-viewer-addon' ),
		'counter' => true
	), $atts );

	$lana_downloads_manager_counter = get_option( 'lana_downloads_manager_counter', true );

	if ( ! empty( $a['id'] ) ) {
		$lana_download = get_post( $a['id'] );
	}

	if ( ! empty( $a['file'] ) ) {
		$lana_download = get_page_by_path( $a['file'], OBJECT, 'lana_download' );
	}

	/** check lana download */
	if ( ! isset( $lana_download ) ) {
		return '';
	}

	/** check is post */
	if ( ! is_a( $lana_download, 'WP_Post' ) ) {
		return '';
	}

	/** post title to text */
	if ( '%post_title%' == $a['text'] ) {
		$a['text'] = $lana_download->post_title;
	}

	$file_url = get_post_meta( $lana_download->ID, 'lana_download_file_url', true );

	list( $file_path ) = lana_downloads_manager_parse_file_path( $file_url );

	if ( empty( $file_path ) ) {
		return '';
	}

	/**
	 * Check pdf
	 */
	$file_ext = pathinfo( wp_basename( $file_path ), PATHINFO_EXTENSION );

	/** check ext */
	if ( 'pdf' != $file_ext ) {
		return '';
	}

	$output = '<div class="lana-download-shortcode">';

	/** download button */
	$output .= '<p>';
	$output .= '<a class="btn btn-primary lana-download lana-pdf-view" href="' . esc_attr( lana_downloads_manager_get_download_url( $lana_download->ID ) ) . '" role="button" target="_blank">';
	$output .= esc_html( $a['text'] ) . ' ';

	/** counter */
	if ( $a['counter'] && $lana_downloads_manager_counter ) {
		$output .= '<span class="badge">';
		$output .= lana_downloads_manager_get_download_count( $lana_download->ID );
		$output .= '</span>';
	}

	$output .= '</a>';
	$output .= '</p>';

	$output .= '</div>';

	return $output;
}

add_shortcode( 'lana_view_pdf', 'lana_view_pdf_instead_of_download_shortcode' );

/**
 * Lana Downloads Manager
 * link target
 *
 * @param $download_id
 */
function lana_downloads_manager_pdf_viewer_addon_link_target( $download_id ) {

	$file_url = get_post_meta( $download_id, 'lana_download_file_url', true );

	if ( empty( $file_url ) ) {
		return '_self';
	}

	list( $file_path, $remote_file, $local_file ) = lana_downloads_manager_parse_file_path( $file_url );

	if ( empty( $file_path ) ) {
		return '_self';
	}

	/**
	 * get file ext
	 * @var string $file_ext
	 */
	$file_ext = pathinfo( wp_basename( $file_path ), PATHINFO_EXTENSION );

	/** is pdf */
	if ( 'pdf' == $file_ext ) {
		return '_blank';
	}

	/** not pdf */
	return '_self';
}